/** @file   weaponguide.cpp
 * @brief   Implementation of WeaponGuide - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "weaponguide.h"
#include "gfxpool.h"
#include "datafiles/datafile_wguide.h"
#include "soundsamples.h"
using namespace eng2d;
using std::string;

namespace WeWantWar {


///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
WeaponGuide::WeaponGuide() :
  m_currentPage( PAGE_CROWBAR ),
  m_lastkey( KEY_ESC )
{
  m_pGfx = load_datafile( "gfx/weaponguide/wguide.dat" );
}



/** Destructor
 */
WeaponGuide::~WeaponGuide()
{
  if ( m_pGfx != 0 ) {
    unload_datafile( m_pGfx );
    m_pGfx = 0;
  }
}




///
/// Public methods
/// ==============

/** Sets the page being drawn
 */
void WeaponGuide::setPage( int pageid )
{
  m_currentPage = pageid;
}



/** Updates this helppage
 */
int WeaponGuide::update()
{
  if ( key[m_lastkey] ) {
    return 0;
  }
  
  m_lastkey = KEY_O;
  if ( key[KEY_LEFT] ) {
    m_lastkey = KEY_LEFT;
    m_currentPage -= 1;
    if ( m_currentPage < 0 ) {
      m_currentPage = PAGE_NUMOF - 1;
    }
    Sound::playSample( SMP_MENUITEM, false );
    
  } else if ( key[KEY_RIGHT ] ) {
    m_lastkey = KEY_RIGHT;
    m_currentPage += 1;
    if ( m_currentPage >= PAGE_NUMOF ) {
      m_currentPage = 0;
    }
    Sound::playSample( SMP_MENUITEM, false );
    
  } else if ( key[KEY_ESC] || key[KEY_SPACE] ) {
    Sound::playSample( SMP_MENUSELECT, false );
    return 1;
  }
  return 0;
}



/** Draws current helppage
 */
void WeaponGuide::redraw( BITMAP* pB )
{
  if ( m_pGfx == 0 ) {
    return;
  }
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;
  blit( pBG, pB, 0,0, x,y, pBG->w, pBG->h );
  
  switch ( m_currentPage ) {
    case ( PAGE_CROWBAR ): {
      this->drawCrowbar( pB );
      break;
    }
    case ( PAGE_RIFLE ): {
      this->drawRifle( pB );
      break;
    }
    case ( PAGE_SHOTGUN ): {
      this->drawShotgun( pB );
      break;
    }
    case ( PAGE_FLAMETHROWER ): {
      this->drawFlamethrower( pB );
      break;
    }
    case ( PAGE_MINIGUN ): {
      this->drawMinigun( pB );
      break;
    }
    case ( PAGE_UZI ): {
      this->drawUzi( pB );
      break;
    }
    case ( PAGE_GRENADE ): {
      this->drawGrenade( pB );
      break;
    }
  }
}




///
/// Getter methods
/// ==============




///
/// Private or Protected methods
/// ============================

/** Draws the crowbar
 */
void WeaponGuide::drawCrowbar( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;
  
  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_CROWBAR].dat );
  draw_sprite( pB, pS, 17+x,14+y );
  
  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "crowbar" );
  
  this->printTwoStrings( "range ", "low", 20+x,110+y, out );
  this->printTwoStrings( "firepower ", "medium", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "1", 20+x,150+y, out );
}



void WeaponGuide::drawRifle( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;

  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_RIFLE].dat );
  draw_sprite( pB, pS, 17+x,14+y );

  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "rifle" );

  this->printTwoStrings( "range ", "high", 20+x,110+y, out );
  this->printTwoStrings( "firepower ", "low", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "2", 20+x, 150+y, out );
}



void WeaponGuide::drawShotgun( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;

  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_SHOTGUN].dat );
  draw_sprite( pB, pS, 17+x,14+y );

  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "shotgun" );

  this->printTwoStrings( "range ", "medium", 20+x,110+y, out );
  this->printTwoStrings( "firepower ", "medium", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "3", 20+x,150+y, out );
}



void WeaponGuide::drawFlamethrower( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;

  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_FLAME].dat );
  draw_sprite( pB, pS, 17+x,14+y );

  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "flamethrower" );

  this->printTwoStrings( "range ", "low", 20+x,110+y, out );
  this->printTwoStrings( "firepower ", "high", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "4", 20+x,150+y, out );
}



void WeaponGuide::drawMinigun( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;

  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_MINIGUN].dat );
  draw_sprite( pB, pS, 17+x,14+y );

  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "minigun" );

  this->printTwoStrings( "range ", "high", 20+x,110+y, out );
  this->printTwoStrings( "firepower ", "high", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "5", 20+x,150+y, out );
}



void WeaponGuide::drawUzi( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;

  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_UZI].dat );
  draw_sprite( pB, pS, 17+x,14+y );

  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "uzi" );

  this->printTwoStrings( "range ", "high", 20+x,110+y, out );
  this->printTwoStrings( "firepower ", "medium", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "6", 20+x,150+y, out );
}



void WeaponGuide::drawGrenade( BITMAP* pB )
{
  BITMAP* pBG = reinterpret_cast<BITMAP*>( m_pGfx[WG_BG].dat );
  int x = 320 - ( pBG->w / 2 );
  int y = 480 - pBG->h;

  BITMAP* pS = reinterpret_cast<BITMAP*>( m_pGfx[WG_GRENADE].dat );
  draw_sprite( pB, pS, 17+x,14+y );

  TPrinter out( pB );
  out.font( GfxPool::pTitleFont, "normal30" );
  out.pos( 110+x, 15+y );
  out.print( "grenade" );

  this->printTwoStrings( "firepower ", "very high", 20+x,130+y, out );
  this->printTwoStrings( "keyboard ", "f", 20+x,150+y, out );
  this->printTwoStrings( "detonate ", "right mouse b", 20+x,170+y, out );
}


void WeaponGuide::printTwoStrings( const string& str1, const string& str2,
                                   int x, int y, TPrinter& rOut )
{
  rOut.pos( x, y );
  rOut.font( GfxPool::pTitleFont, "menuitems" );
  rOut.print( str1 );
  
  rOut.font( GfxPool::pTitleFont, "menuitems_sel" );
  rOut.print( str2 );
}



} // end of namespace
